#[cfg(all(feature = "music", not(feature = "32k")))]
use sdl2::audio::AudioCallback;

#[cfg(all(feature = "music", not(feature = "32k")))]
use crate::vendored::micromod::MmC2r;

#[cfg(all(feature = "music", not(feature = "32k")))]
pub struct Music {
  pub pcm: Vec<i16>,
  pub offset: usize,
  pub playing: bool,
}

#[cfg(all(feature = "music", not(feature = "32k")))]
impl AudioCallback for Music {
  type Channel = f32;

  fn callback(&mut self, out_f: &mut [f32]) {
    if self.playing {
      let len = out_f.len();
      let end_of_slice = (self.offset + len).min(self.pcm.len() - 1);
      let slice = &self.pcm[self.offset..end_of_slice];
      let len = slice.len();
      pcm_to_f32pcm(&slice, out_f, len);
      self.offset += len;
      if end_of_slice == self.pcm.len() - 1 {
        self.offset = 0;
        #[cfg(not(feature = "loop_music"))]
        {
          self.playing = false;
        }
      }
    } else {
      for i in 0..out_f.len() - 1 {
        out_f[i] = 0.0;
      }
    }
  }
}

#[cfg(all(feature = "music", not(feature = "32k")))]
pub fn mmc2r_to_pcm(state: &mut MmC2r) -> Vec<i16> {
  let mut destination = Vec::<i16>::new();
  loop {
    let mut out = [0; 4096];
    if !state.get_audio(&mut out, 2048) {
      break;
    }
    for sample in out {
      destination.push(sample);
    }
  }
  #[cfg(not(feature = "stereo"))]
  let destination = {
    let stereo = destination;
    let mut mono = Vec::<i16>::new();
    let mut is_first_stereo_pair = true;
    let mut stereo_pair_val = 0 as i16;
    for sample in stereo {
      if is_first_stereo_pair {
        stereo_pair_val = sample;
      } else {
        mono.push(((sample as i32 + stereo_pair_val as i32) / 2) as i16);
      }
      is_first_stereo_pair = !is_first_stereo_pair;
    }
    mono
  };
  destination
}

#[cfg(all(feature = "music", not(feature = "32k")))]
pub fn pcm_to_f32pcm(input: &[i16], output: &mut [f32], len: usize) {
  for i in 0..len {
    output[i] = f32::from(input[i]) / f32::from(0x7FFF as i16);
  }
}
